#!/bin/bash
DOCKER_BASE=/srv/docker

mkdir -p ${DOCKER_BASE}/traefik/container.conf

cat > ${DOCKER_BASE}/traefik/container.conf/docker-compose.yml <<EOF
version: '3.7'

services:
  traefik:
    image: traefik:1.7-alpine
    networks:
      - system_traefik
    environment:
      - LC_ALL=C.UTF-8
      - TZ=Europe/Berlin
    labels:
      - traefik.enable=true
      - traefik.backend=traefik
      - traefik.port=8080
    ports:
      - "80:80"
      - "443:443"
      - "8080:8080"
    restart: always
    volumes:
      - "./config/:/etc/traefik/"
      - "/var/run/docker.sock:/var/run/docker.sock:ro"

networks:
  system_traefik:
    external: true
EOF
ln -s container.conf/docker-compose.yml ${DOCKER_BASE}/traefik/

cat > ${DOCKER_BASE}/traefik/container.conf/production.yml <<EOF
version: '3.7'

services:

  traefik:
    logging:
      options:
        max-size: "100M"
        max-file: "10"
    labels:
      - traefik.frontend.rule=Host:host.test.org;PathPrefixStrip:/traefik
      - com.centurylinklabs.watchtower.enable=true
EOF

cat > ${DOCKER_BASE}/traefik/container.conf/traefik.service <<EOF
[Unit]
Description=Traefik Proxy Service
After=network.target docker.service
Requires=docker.service

[Service]
Type=oneshot
RemainAfterExit=yes

Environment="WORK_DIR=/srv/docker/traefik/"
WorkingDirectory=/srv/docker/traefik/
ExecStartPre=/bin/bash -c "/usr/bin/docker network inspect system_traefik &>/dev/null || /usr/bin/docker network create --driver bridge system_traefik"
ExecStartPre=-/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" down
ExecStart=/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" up -d
ExecStop=/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" down

[Install]
WantedBy=docker.service
EOF
ln -s ${DOCKER_BASE}/traefik/container.conf/traefik.service /etc/systemd/system/

mkdir -p ${DOCKER_BASE}/traefik/config

cat > ${DOCKER_BASE}/traefik/config/traefik.toml <<EOF
logLevel = "DEBUG"
defaultEntryPoints = ["http", "https"]

# WEB interface of Traefik - it will show web page with overview of frontend and backend configurations 
[web]
address = ":8080"
  [web.auth.basic]
  users = ["admin:$apr1$AAbCdQpX$ajolS9mMfKRG.lqcY/uXU/"]

# Connection to docker host system (docker.sock)
[docker]
domain = "test.org"
watch = true
# This will hide all docker containers that don't have explicitly  
# set label to "enable"
exposedbydefault = false

# Force HTTPS
[entryPoints]
  [entryPoints.http]
  address = ":80"
    [entryPoints.http.redirect]
    entryPoint = "https"
  [entryPoints.https]
  address = ":443"
    [entryPoints.https.tls]
    minVersion = "VersionTLS12"
  
# Let's encrypt configuration
[acme]
  email="ssladmin@test.org"
  storage="/etc/traefik/acme.json"
  entryPoint="https"
  acmeLogging=true
  onDemand=false
  OnHostRule=true

[acme.httpChallenge]
  entryPoint = "http"
EOF

systemctl daemon-reload && systemctl enable traefik && systemctl start traefik
