#!/bin/bash
DOCKER_BASE="${DOCKER_BASE:-/srv/docker}"
TRAEFIK_DIR="${TRAEFIK_DIR:-traefik}"

mkdir -p ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf

touch ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf/.env
ln -s container.conf/.env ${DOCKER_BASE}/${TRAEFIK_DIR}/

cat > ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf/docker-compose.yml <<EOF
# Inspired by https://containo.us/blog/traefik-2-0-docker-101-fc2893944b9d/

version: '3'

services:
  traefik:
    image: traefik:2.11
    command:
      - --entrypoints.web.address=:80
      - --entrypoints.websecure.address=:443
      - "--log.level=INFO"
      - --providers.docker=true
      - --providers.docker.exposedbydefault=false
      - --api=true
      - --certificatesresolvers.default.acme.caserver=\${LEAPI:-https://acme-v02.api.letsencrypt.org/directory}
      - --certificatesresolvers.default.acme.email=\${LEMAIL:-ssladmin@test.org}
      - --certificatesresolvers.default.acme.storage=/etc/traefik/acme.json
      - --certificatesresolvers.default.acme.tlschallenge=true
      - --providers.file.filename=/etc/traefik/traefik_providers.yaml
      - --providers.file.watch=true
    logging:
      options:
        max-size: "100M"
        max-file: "10"
    networks:
      - system_traefik
    environment:
      - LC_ALL=C.UTF-8
      - TZ=Europe/Berlin
    labels:
      # Enable Traefik for it's own backend
      - traefik.enable=true
      # Dashboard
      - traefik.http.routers.traefik.rule=Host(\`traefik.test.org\`)
      - traefik.http.routers.traefik.entrypoints=websecure
      - traefik.http.routers.traefik.tls=true
      - traefik.http.routers.traefik.tls.certresolver=default
      - traefik.http.routers.traefik.service=api@internal
      # Basic auth for dashboard
      - traefik.http.routers.traefik.middlewares=authtraefik@docker,default-security-headers@file
      # middleware authtraefik
      - traefik.http.middlewares.authtraefik.basicauth.users=\${DASHBOARD_USERS:-admin:\$\$apr1\$\$AAbCdQpX\$\$ajelS9mMisKRG.lqcY/uXU/} # user/password
    ports:
      - "80:80"
      - "443:443"
    restart: always
    volumes:
      - "./config/:/etc/traefik/"
      - "/var/run/docker.sock:/var/run/docker.sock:ro"

networks:
  system_traefik:
    external: true
EOF
ln -s container.conf/docker-compose.yml ${DOCKER_BASE}/${TRAEFIK_DIR}/

cat > ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf/production.yml <<EOF
version: '3'

services:

  traefik:
    labels:
      # Allow watchtower to update this image
      - com.centurylinklabs.watchtower.enable=true
      # See https://docs.traefik.io/migration/v1-to-v2/#strip-and-rewrite-path-prefixes
      - traefik.http.routers.traefik.rule=Host(\`$(hostname -f)\`) && (PathPrefix(\`/traefik\`) || PathPrefix(\`/api\`))
      # Redefine middleware for router 'traefik' as we add more middlewares
      - traefik.http.routers.traefik.middlewares=authtraefik@docker,traefik-dashboard-stripprefix@file,default-security-headers@file
EOF

cat > ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf/traefik.service <<EOF
[Unit]
Description=Traefik Proxy Service
After=network.target docker.service
Requires=docker.service

[Service]
Type=oneshot
RemainAfterExit=yes

Environment="WORK_DIR=/srv/docker/traefik/"
WorkingDirectory=/srv/docker/traefik/
ExecStartPre=/bin/bash -c "/usr/bin/docker network inspect system_traefik &>/dev/null || /usr/bin/docker network create --driver bridge system_traefik"
ExecStartPre=-/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" down
ExecStart=/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" up -d
ExecStop=/usr/local/bin/docker-compose -f "\${WORK_DIR}/docker-compose.yml" -f "\${WORK_DIR}/container.conf/production.yml" down

[Install]
WantedBy=docker.service
EOF
ln -s ${DOCKER_BASE}/${TRAEFIK_DIR}/container.conf/traefik.service /etc/systemd/system/

mkdir -p ${DOCKER_BASE}/${TRAEFIK_DIR}/config

cat > ${DOCKER_BASE}/${TRAEFIK_DIR}/config/traefik_providers.yaml <<EOF
---
tls:
  options:
    default:
      minVersion: VersionTLS12
      sniStrict: true
      cipherSuites:
        # TLS 1.2 cipher suites.
        - TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256
        - TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384
        - TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305
        # IE 11 and Safari < 9 + iOS <9, OSX < 10.11
        - TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA
        # TLS 1.3 cipher suites.
        - TLS_AES_128_GCM_SHA256
        - TLS_AES_256_GCM_SHA384
        - TLS_CHACHA20_POLY1305_SHA256
        # TLS_FALLBACK_SCSV isn't a standard cipher suite but an indicator that the client is doing version fallback. See RFC 7507.
        - TLS_FALLBACK_SCSV
    curvePreferences:
      - CurveP521
      - CurveP384

http:
  middlewares:
    redirect-web-to-websecure:
      redirectScheme:
        scheme: https
        permanent: true
    default-security-headers:
      headers:
        accessControlAllowMethods:
          - GET
          - POST
          - DELETE
          - OPTIONS
        accessControlAllowOriginList: ["<origin>"]
        accessControlMaxAge: 100
        browserXssFilter: true
        contentTypeNosniff: true
        forceSTSHeader: true
#        frameDeny: true
#        sslRedirect: true
        stsIncludeSubdomains: true
        stsPreload: true
        stsSeconds: 315360000
#        contentSecurityPolicy: "default-src 'self' 'unsafe-inline';script-src 'self' 'unsafe-inline' 'unsafe-eval';img-src 'self' data:;font-src 'self' data:;connect-src 'self' ws: wss:"
#        customRequestHeaders:
#          X-Frame-Options: "SAMEORIGIN"
#        customFrameOptionsValue: "SAMEORIGIN"
        referrerPolicy: "same-origin"
        permissionsPolicy: "vibrate=(self)"
    traefik-dashboard-stripprefix:
      stripPrefix:
        prefixes:
          - "/traefik"
  services:
    redirect-dummy:
      loadBalancer:
        servers:
          - url: ""
  routers:
    # global redirect to https
    # per domain see https://doc.traefik.io/traefik/migration/v1-to-v2/#http-to-https-redirection-is-now-configured-on-routers
    web-to-websecure:
      rule: "hostregexp(\`{host:.+}\`)"
      service: "redirect-dummy@file"
      entryPoints:
        - "web"
      middlewares:
        - redirect-web-to-websecure@file
EOF

systemctl daemon-reload && systemctl enable traefik && systemctl start traefik
